% Snells Law And Critical Angle Reflection Example
%
% This example demonstrates Snells law and critical angle reflection
%
% author: Bradley Treeby
% date: 10th December 2009
% last update: 20th January 2010
%  
% This function is part of the k-Wave Toolbox (http://www.k-wave.org)
% Copyright (C) 2009, 2010 Bradley Treeby and Ben Cox

% This file is part of k-Wave. k-Wave is free software: you can
% redistribute it and/or modify it under the terms of the GNU Lesser
% General Public License as published by the Free Software Foundation,
% either version 3 of the License, or (at your option) any later version.
% 
% k-Wave is distributed in the hope that it will be useful, but WITHOUT ANY
% WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
% FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
% more details. 
% 
% You should have received a copy of the GNU Lesser General Public License
% along with k-Wave. If not, see <http://www.gnu.org/licenses/>. 

clear all;

% =========================================================================
% SIMULATION
% =========================================================================

% create the computational grid
Nz = 128;           % number of pixels in the x (column) direction
Nx = Nz;            % number of pixels in the z (row) direction
dz = 50e-3/Nz;    	% pixel width [m]
dx = dz;            % pixel height [m]
kgrid = makeGrid(Nx, dx, Nz, dz);

% define the properties of a layered propagation medium
c0 = 1500;
medium_mulp = 2;
medium.sound_speed = c0*ones(Nz, Nx);  % [m/s]
medium.sound_speed(Nz/2:end, :) = medium_mulp*c0;
medium.density = 1000;
medium.alpha_power = 1.5;   % [dB/(MHz^y cm)]
medium.alpha_coeff = 0.75;  % [dB/(MHz^y cm)]

% create the time array
[kgrid.t_array dt] = makeTime(kgrid, medium.sound_speed, 0.3);
kgrid.t_array = 0:dt:1000*dt;

% define a linear element transducer with an odd number of elements
num_elements = 61;
source.p_mask = zeros(Nz, Nx);
x_offset = 20;
z_offset = 25;
start_index = Nz/2 - round(num_elements/2) + 1 - x_offset;
source.p_mask(z_offset, start_index:start_index + num_elements - 1) = 1;

% define the properties of the tone burst using a geometrically steered
% transmit pulse offset
steering_angle = 20;
element_spacing = dz;
element_index = -(num_elements - 1)/2:(num_elements - 1)/2;
tone_burst_offset = 200 + element_spacing*element_index*sin(steering_angle*pi/180)/(c0*dt);
tone_burst_freq = 1e6;
tone_burst_cycles = 8;
tone_burst_signal_length = length(kgrid.t_array);
sampling_freq = 1/dt;
source.p = toneBurst(sampling_freq, tone_burst_freq, tone_burst_signal_length, tone_burst_offset, tone_burst_cycles);

% define the sensor to be identical to the source
sensor.mask = source.p_mask;

% run the simulation
kspaceFirstOrder2D(kgrid, medium, source, sensor);